// Use the exposed electron API from preload script
const ipcRenderer = window.electron;

class Logger {
  constructor() {
    this.ready = true;
    console.log("Logger instance created");
  }

  // Get source name from current file
  getSource() {
    try {
      const stack = new Error().stack;
      const callerFile = stack.split("\n")[3].trim().split(" ")[1];
      const fileName = callerFile
        .split("/")
        .pop()
        .split("\\")
        .pop()
        .split(":")[0];
      return fileName || "RENDERER";
    } catch (e) {
      return "RENDERER";
    }
  }

  // Log methods with auto-fallback
  info(message) {
    console.info(message);
    try {
      ipcRenderer
        .invoke("log-message", "info", this.getSource(), message)
        .catch(() => {
          /* Silent fallback */
        });
    } catch (err) {
      // Silent fallback - already logged to console
    }
  }

  debug(message) {
    console.debug(message);
    try {
      ipcRenderer
        .invoke("log-message", "debug", this.getSource(), message)
        .catch(() => {
          /* Silent fallback */
        });
    } catch (err) {
      // Silent fallback - already logged to console
    }
  }

  warn(message) {
    console.warn(message);
    try {
      ipcRenderer
        .invoke("log-message", "warn", this.getSource(), message)
        .catch(() => {
          /* Silent fallback */
        });
    } catch (err) {
      // Silent fallback - already logged to console
    }
  }

  error(message, error = null) {
    let errorMsg = message;

    if (error) {
      console.error(message, error);
      if (error instanceof Error) {
        errorMsg += `: ${error.message}\n${error.stack || ""}`;
      } else {
        errorMsg += `: ${String(error)}`;
      }
    } else {
      console.error(message);
    }

    try {
      ipcRenderer
        .invoke("log-message", "error", this.getSource(), errorMsg)
        .catch(() => {
          /* Silent fallback */
        });
    } catch (err) {
      // Silent fallback - already logged to console
    }
  }

  // Get log file path with fallback
  async getLogPath() {
    try {
      return await ipcRenderer.invoke("get-log-path").catch(() => null);
    } catch (err) {
      console.error("Failed to get log path:", err);
      return null;
    }
  }
}

// Export a singleton
module.exports = new Logger();
